<?php

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class ICP_Backup {
    
    private static $backup_dir = null;
    
    /**
     * Get the backup directory path
     */
    public static function get_backup_dir() {
        if (self::$backup_dir === null) {
            $upload_dir = wp_upload_dir();
            self::$backup_dir = $upload_dir['basedir'] . '/icp-backups';
        }
        return self::$backup_dir;
    }
    
    /**
     * Get the backup directory URL
     */
    public static function get_backup_url() {
        $upload_dir = wp_upload_dir();
        return $upload_dir['baseurl'] . '/icp-backups';
    }
    
    /**
     * Create backup directory structure if it doesn't exist
     */
    public static function ensure_backup_dir() {
        $backup_dir = self::get_backup_dir();
        
        if (!file_exists($backup_dir)) {
            wp_mkdir_p($backup_dir);
            
            // Create .htaccess to protect backup files
            $htaccess_content = "# Protect backup files\n";
            $htaccess_content .= "Order deny,allow\n";
            $htaccess_content .= "Deny from all\n";
            $htaccess_content .= "<Files ~ \"\\.(jpg|jpeg|png|gif|webp)$\">\n";
            $htaccess_content .= "    Allow from all\n";
            $htaccess_content .= "</Files>\n";
            
            file_put_contents($backup_dir . '/.htaccess', $htaccess_content);
            
            // Create index.php to prevent directory listing
            file_put_contents($backup_dir . '/index.php', '<?php // Silence is golden');
        }
        
        return file_exists($backup_dir) && is_writable($backup_dir);
    }
    
    /**
     * Generate backup file path for an attachment
     */
    public static function generate_backup_path($attachment_id, $original_file_path) {
        $backup_dir = self::get_backup_dir();
        $file_info = pathinfo($original_file_path);
        
        // Create year/month structure like WordPress uploads
        $date = get_the_date('Y/m', $attachment_id);
        if (!$date) {
            $date = date('Y/m');
        }
        
        $backup_subdir = $backup_dir . '/' . $date;
        wp_mkdir_p($backup_subdir);
        
        // Generate unique filename to avoid conflicts
        $filename = $file_info['filename'] . '-' . $attachment_id . '-original.' . $file_info['extension'];
        
        return $backup_subdir . '/' . $filename;
    }
    
    /**
     * Create backup of original image before compression
     */
    public static function backup_original($attachment_id, $original_file_path) {
        if (!file_exists($original_file_path)) {
            return array(
                'success' => false,
                'error' => 'Original file not found: ' . $original_file_path
            );
        }
        
        if (!self::ensure_backup_dir()) {
            return array(
                'success' => false,
                'error' => 'Could not create or access backup directory'
            );
        }
        
        $backup_path = self::generate_backup_path($attachment_id, $original_file_path);
        
        // Copy original file to backup location
        if (copy($original_file_path, $backup_path)) {
            return array(
                'success' => true,
                'backup_path' => $backup_path
            );
        } else {
            return array(
                'success' => false,
                'error' => 'Failed to copy file to backup location'
            );
        }
    }
    
    /**
     * Restore original image from backup
     */
    public static function restore_original($attachment_id, $backup_path, $current_file_path) {
        if (!file_exists($backup_path)) {
            return array(
                'success' => false,
                'error' => 'Backup file not found: ' . $backup_path
            );
        }
        
        if (!file_exists($current_file_path)) {
            return array(
                'success' => false,
                'error' => 'Current file not found: ' . $current_file_path
            );
        }
        
        // Create backup of current compressed file (in case restore fails)
        $temp_backup = $current_file_path . '.temp_backup';
        if (!copy($current_file_path, $temp_backup)) {
            return array(
                'success' => false,
                'error' => 'Could not create temporary backup of current file'
            );
        }
        
        // Restore original file
        if (copy($backup_path, $current_file_path)) {
            // Remove temporary backup
            unlink($temp_backup);
            
            // Regenerate WordPress thumbnails
            self::regenerate_thumbnails($attachment_id);
            
            return array(
                'success' => true,
                'message' => 'Original image restored successfully'
            );
        } else {
            // Restore failed, put back the compressed version
            copy($temp_backup, $current_file_path);
            unlink($temp_backup);
            
            return array(
                'success' => false,
                'error' => 'Failed to restore original file'
            );
        }
    }
    
    /**
     * Regenerate WordPress thumbnails for an attachment
     */
    private static function regenerate_thumbnails($attachment_id) {
        $file_path = get_attached_file($attachment_id);
        
        if (!$file_path || !file_exists($file_path)) {
            return false;
        }
        
        // Include WordPress image functions
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        
        // Generate new metadata and thumbnails
        $metadata = wp_generate_attachment_metadata($attachment_id, $file_path);
        
        if ($metadata) {
            wp_update_attachment_metadata($attachment_id, $metadata);
            return true;
        }
        
        return false;
    }
    
    /**
     * Delete backup file
     */
    public static function delete_backup($backup_path) {
        if (file_exists($backup_path)) {
            return unlink($backup_path);
        }
        return true; // Already deleted
    }
    
    /**
     * Get backup file size
     */
    public static function get_backup_size($backup_path) {
        if (file_exists($backup_path)) {
            return filesize($backup_path);
        }
        return 0;
    }
    
    /**
     * Check if backup file exists and is readable
     */
    public static function backup_exists($backup_path) {
        return file_exists($backup_path) && is_readable($backup_path);
    }
    
    /**
     * Clean up old backup files (optional maintenance function)
     */
    public static function cleanup_old_backups($days_old = 30) {
        $backup_dir = self::get_backup_dir();
        
        if (!is_dir($backup_dir)) {
            return 0;
        }
        
        $cutoff_time = time() - ($days_old * 24 * 60 * 60);
        $deleted_count = 0;
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($backup_dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );
        
        foreach ($iterator as $file) {
            if ($file->isFile() && $file->getMTime() < $cutoff_time) {
                if (unlink($file->getPathname())) {
                    $deleted_count++;
                }
            }
        }
        
        return $deleted_count;
    }
    
    /**
     * Get total size of backup directory
     */
    public static function get_total_backup_size() {
        $backup_dir = self::get_backup_dir();
        
        if (!is_dir($backup_dir)) {
            return 0;
        }
        
        $total_size = 0;
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($backup_dir, RecursiveDirectoryIterator::SKIP_DOTS)
        );
        
        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $total_size += $file->getSize();
            }
        }
        
        return $total_size;
    }
}
